<?php
defined('ABSPATH') || exit;

/**
 * Receives and validates signed webhooks from Agora OS.
 *
 * Endpoint: POST /wp-json/agora/v1/webhook
 *
 * Agora signs payloads with HMAC-SHA256 using the store's
 * webhook secret (stored as agora_webhook_secret option).
 * The signature is sent in the X-Agora-Signature-256 header
 * as "sha256=<hex>".
 *
 * Currently handled events:
 *   - order.fulfilled: Agora reports an order was delivered to the agent
 */
class Agora_Webhook {

    public function __construct() {
        add_action('rest_api_init', [$this, 'register_route']);
    }

    public function register_route(): void {
        register_rest_route('agora/v1', '/webhook', [
            'methods'             => 'POST',
            'callback'            => [$this, 'handle'],
            'permission_callback' => '__return_true', // Signature validation done inside
        ]);
    }

    public function handle(\WP_REST_Request $request): \WP_REST_Response {
        $secret = get_option('agora_webhook_secret');
        if (!$secret) {
            return new \WP_REST_Response(['error' => 'Webhook not configured'], 501);
        }

        // Validate HMAC signature
        $raw_body  = $request->get_body();
        $signature = $request->get_header('X-Agora-Signature-256');

        if (!$this->verify_signature($raw_body, $signature, $secret)) {
            return new \WP_REST_Response(['error' => 'Invalid signature'], 401);
        }

        $payload = json_decode($raw_body, true);
        if (!is_array($payload) || empty($payload['event'])) {
            return new \WP_REST_Response(['error' => 'Invalid payload'], 400);
        }

        $this->dispatch($payload['event'], $payload['data'] ?? []);

        return new \WP_REST_Response(['ok' => true], 200);
    }

    private function verify_signature(string $body, ?string $header, string $secret): bool {
        if (!$header || !str_starts_with($header, 'sha256=')) return false;

        $provided = substr($header, 7);
        $expected = hash_hmac('sha256', $body, $secret);

        return hash_equals($expected, $provided);
    }

    private function dispatch(string $event, array $data): void {
        switch ($event) {
            case 'order.status_updated':
                $this->on_order_status_updated($data);
                break;
            default:
                error_log('[Agora] Unknown webhook event: ' . $event);
        }
    }

    /**
     * Agora notifies the store that an order's status changed on the Agora side.
     * Primarily useful for logging/admin visibility.
     */
    private function on_order_status_updated(array $data): void {
        $wc_order_id  = isset($data['wc_order_id']) ? (int) $data['wc_order_id'] : 0;
        $agora_status = sanitize_text_field($data['status'] ?? '');

        if (!$wc_order_id || !$agora_status) return;

        $order = wc_get_order($wc_order_id);
        if (!$order) return;

        $order->add_order_note(
            /* translators: %s: Agora order status */
            sprintf(__('Agora status updated: %s', 'agora-agent-commerce'), esc_html($agora_status))
        );
    }
}
