<?php
defined('ABSPATH') || exit;

/**
 * Serves the Agora agent manifest at /.well-known/agent.json
 *
 * This allows AI agent frameworks to discover this store's
 * Agora capabilities via the emerging agentic-web standard.
 *
 * Note: Only exposes data if the store is connected to Agora
 * (agora_merchant_id option is set). Returns 404 otherwise.
 */
class Agora_Manifest {

    public function __construct() {
        add_action('init', [$this, 'register_rewrite_rule']);
        add_filter('query_vars', [$this, 'add_query_var']);
        add_action('template_redirect', [$this, 'handle_request']);
    }

    public function register_rewrite_rule(): void {
        add_rewrite_rule('^\.well-known/agent\.json$', 'index.php?agora_manifest=1', 'top');
        // Flush on plugin activation; no need to flush on every request
    }

    public function add_query_var(array $vars): array {
        $vars[] = 'agora_manifest';
        return $vars;
    }

    public function handle_request(): void {
        if (!get_query_var('agora_manifest')) return;

        $merchant_id = get_option('agora_merchant_id');
        if (!$merchant_id) {
            status_header(404);
            exit;
        }

        $manifest = [
            'schema_version'    => '1.0',
            'name'              => get_bloginfo('name'),
            'description'       => get_bloginfo('description') ?: null,
            'platform'          => 'woocommerce',
            'agora_merchant_id' => $merchant_id,
            'capabilities'      => ['product_search', 'purchase'],
            'commerce_endpoint' => AGORA_API_BASE_URL . '/commerce',
            'currency'          => get_woocommerce_currency(),
            'locale'            => get_locale(),
        ];

        header('Content-Type: application/json; charset=utf-8');
        header('Cache-Control: public, max-age=3600');
        header('X-Robots-Tag: noindex');

        echo wp_json_encode($manifest, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        exit;
    }

}
