<?php
defined('ABSPATH') || exit;

/**
 * Handles Agora-originated WooCommerce orders.
 *
 * Responsibilities:
 * - Marks WC orders that came from Agora with _agora_order_id meta
 * - Notifies Agora API when order status changes (best-effort)
 * - Adds Agora order details to WC admin order view
 */
class Agora_Checkout {

    private Agora_API $api;

    public function __construct() {
        $this->api = new Agora_API();

        // Add Agora order info to WC admin order view
        add_action('woocommerce_admin_order_data_after_order_details', [$this, 'render_order_meta']);

        // Notify Agora when a tagged order's status changes
        add_action('woocommerce_order_status_changed', [$this, 'on_order_status_changed'], 10, 3);
    }

    /**
     * Called by Agora_API after successfully placing a WC order.
     * Stores the Agora order ID on the WC order for traceability.
     */
    public function tag_order(int $wc_order_id, string $agora_order_id): void {
        $order = wc_get_order($wc_order_id);
        if (!$order) return;

        $order->update_meta_data('_agora_order_id', $agora_order_id);
        $order->update_meta_data('_agora_placed_at', current_time('mysql'));
        $order->save();
    }

    /**
     * Display Agora metadata in the WC admin order screen.
     */
    public function render_order_meta(\WC_Order $order): void {
        $agora_id = $order->get_meta('_agora_order_id');
        if (!$agora_id) return;

        echo '<p class="form-field form-field-wide">';
        echo '<strong>' . esc_html__('Agora Order ID', 'agora-agent-commerce') . ':</strong> ';
        echo '<code>' . esc_html($agora_id) . '</code>';
        echo '</p>';
    }

    /**
     * Notify Agora when a tagged WC order changes status.
     *
     * Maps WC statuses to Agora fulfillment states:
     *   completed  → notify fulfilled
     *   cancelled  → notify cancelled
     *   refunded   → notify refunded
     */
    public function on_order_status_changed(int $order_id, string $old_status, string $new_status): void {
        $order = wc_get_order($order_id);
        if (!$order) return;

        $agora_order_id = $order->get_meta('_agora_order_id');
        if (!$agora_order_id) return;

        $notify_statuses = ['completed', 'cancelled', 'refunded'];
        if (!in_array($new_status, $notify_statuses, true)) return;

        // Best-effort notification — log failures but don't block WC
        $result = $this->api->notify_order_status($agora_order_id, $new_status);
        if (is_wp_error($result)) {
            error_log('[Agora] Failed to notify order status change for ' . $agora_order_id . ': ' . $result->get_error_message());
        }
    }
}
