<?php
defined('ABSPATH') || exit;

/**
 * Handles WooCommerce → Agora product catalog sync.
 * Uses chunks of 100 products to stay within API and CF Workers limits.
 */
class Agora_Catalog {

    private const BATCH_SIZE = 100;

    private Agora_API $api;

    public function __construct() {
        $this->api = new Agora_API();
    }

    /**
     * Sync a single product (called from real-time hooks).
     */
    public function sync_product(int $product_id): void {
        if (!get_option('agora_merchant_id')) return; // Not connected

        $product = wc_get_product($product_id);
        if (!$product || $product->get_status() !== 'publish') return;

        $normalized = $this->normalize_product($product);
        if (!$normalized) return;

        $result = $this->api->batch_products([$normalized]);
        if (is_wp_error($result)) {
            error_log('[Agora] sync_product failed for ' . $product_id . ': ' . $result->get_error_message());
        }
    }

    /**
     * Full catalog sync — chunks products into batches of 100.
     * Called on activation and nightly via WP-Cron.
     */
    public function full_sync(): void {
        if (!get_option('agora_merchant_id')) return;

        $page = 1;
        $synced = 0;

        do {
            $products = wc_get_products([
                'status'   => 'publish',
                'limit'    => self::BATCH_SIZE,
                'page'     => $page,
                'paginate' => false,
            ]);

            if (empty($products)) break;

            $normalized = array_filter(array_map([$this, 'normalize_product'], $products));

            if (!empty($normalized)) {
                $result = $this->api->batch_products(array_values($normalized));
                if (is_wp_error($result)) {
                    error_log('[Agora] full_sync batch failed (page ' . $page . '): ' . $result->get_error_message());
                    // Continue to next batch rather than aborting entire sync
                } else {
                    $synced += count($normalized);
                }
            }

            $page++;
        } while (count($products) === self::BATCH_SIZE);

        update_option('agora_last_sync', current_time('mysql'));
        update_option('agora_last_sync_count', $synced);
        error_log('[Agora] full_sync complete: ' . $synced . ' products synced');
    }

    /**
     * Convert a WC_Product to Agora's normalized schema.
     * Returns null if the product should not be synced.
     */
    private function normalize_product(WC_Product $product): ?array {
        // Skip products without a price (drafts, grouped products we can't price)
        $price = (float) $product->get_price();
        if ($price <= 0 && $product->get_type() !== 'variable') return null;

        $description = wp_strip_all_tags($product->get_short_description() ?: $product->get_description());
        $description = mb_substr($description, 0, 2000);

        // Map WC categories to the first category name (Google Product Taxonomy mapping is Phase 1)
        $category_ids = $product->get_category_ids();
        $category_name = null;
        if (!empty($category_ids)) {
            $term = get_term($category_ids[0], 'product_cat');
            if ($term && !is_wp_error($term)) {
                $category_name = $term->name;
            }
        }

        // Tags
        $tag_ids = $product->get_tag_ids();
        $tags = [];
        foreach ($tag_ids as $tag_id) {
            $term = get_term($tag_id, 'product_tag');
            if ($term && !is_wp_error($term)) {
                $tags[] = $term->name;
            }
        }

        // Images
        $image_id  = $product->get_image_id();
        $image_url = $image_id ? wp_get_attachment_url($image_id) : null;
        $gallery   = array_filter(array_map('wp_get_attachment_url', $product->get_gallery_image_ids()));

        // Variants (for variable products)
        $variants = [];
        if ($product->is_type('variable')) {
            $variation_ids = $product->get_children();
            foreach ($variation_ids as $variation_id) {
                $variation = wc_get_product($variation_id);
                if (!$variation instanceof WC_Product_Variation) continue;

                $var_price = (float) $variation->get_price();
                if ($var_price <= 0) continue;

                $options = [];
                foreach ($variation->get_variation_attributes() as $attribute => $value) {
                    $clean_key = str_replace('attribute_', '', $attribute);
                    $clean_key = wc_attribute_label($clean_key, $product);
                    $options[$clean_key] = $value;
                }

                $variants[] = [
                    'sku'          => $variation->get_sku() ?: 'var-' . $variation_id,
                    'variation_id' => $variation_id, // Required for WC REST order checkout
                    'title'        => implode(' / ', array_values($options)) ?: 'Default',
                    'price'        => $var_price,
                    'available'    => $variation->is_in_stock(),
                    'options'      => $options,
                ];
            }

            // Use lowest variant price as the base price
            $prices = array_column($variants, 'price');
            $price  = !empty($prices) ? min($prices) : 0;
            if ($price <= 0) return null;
        }

        $regular_price = (float) $product->get_regular_price();

        return [
            'platform_product_id' => (string) $product->get_id(),
            'title'               => mb_substr($product->get_name(), 0, 200),
            'description'         => $description ?: null,
            'category'            => $category_name,
            'tags'                => $tags,
            'price'               => $price,
            'currency'            => get_woocommerce_currency(),
            'compare_at_price'    => ($regular_price > $price) ? $regular_price : null,
            'variants'            => $variants,
            'image_url'           => $image_url ?: null,
            'images'              => array_values($gallery),
            'in_stock'            => $product->is_in_stock(),
            'inventory_count'     => $product->managing_stock() ? $product->get_stock_quantity() : null,
            'source_url'          => get_permalink($product->get_id()),
        ];
    }
}
