<?php
defined('ABSPATH') || exit;

/**
 * Agora API client — handles all HTTP communication with api.agora.sh
 */
class Agora_API {

    private string $base_url;
    private ?string $api_key;

    public function __construct() {
        $this->base_url = defined('AGORA_API_BASE_URL') ? AGORA_API_BASE_URL : 'https://api.agora.sh';
        $this->api_key  = get_option('agora_api_key');
    }

    /**
     * Register this WooCommerce store with Agora.
     * Returns [ merchant_id, agora_api_key ] on success, WP_Error on failure.
     */
    public function register_merchant(array $data): array|WP_Error {
        $response = $this->post('/commerce/merchants/register', $data, false);
        if (is_wp_error($response)) return $response;

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);

        if ($code !== 201) {
            return new WP_Error(
                'agora_register_failed',
                $body['error']['message'] ?? __('Registration failed. Check your WooCommerce credentials.', 'agora-agent-commerce')
            );
        }

        return $body;
    }

    /**
     * Batch upsert products (max 100 per call — chunks handled by Agora_Catalog).
     */
    public function batch_products(array $products): true|WP_Error {
        if (empty($products)) return true;

        $response = $this->post('/commerce/products/batch', ['products' => $products]);
        if (is_wp_error($response)) return $response;

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            return new WP_Error('agora_batch_failed', $body['error']['message'] ?? 'Batch sync failed');
        }

        return true;
    }

    /**
     * Validate an Agora merchant API key.
     * Returns [ merchant_id, name, valid ] on success, WP_Error on failure.
     */
    public function validate_key(string $api_key): array|WP_Error {
        $response = wp_remote_get(
            $this->base_url . '/commerce/merchants/me/validate',
            [
                'headers' => ['Authorization' => 'Bearer ' . $api_key],
                'timeout' => 15,
            ]
        );

        if (is_wp_error($response)) return $response;

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);

        if ($code !== 200) {
            return new WP_Error(
                'agora_invalid_key',
                $body['error']['message'] ?? __('Invalid API key. Check that you copied it correctly from agora.sh.', 'agora-agent-commerce')
            );
        }

        return $body;
    }

    /**
     * Mark this merchant as inactive on Agora (called on disconnect).
     */
    public function deactivate_merchant(string $merchant_id): true|WP_Error {
        $response = $this->post('/commerce/merchants/' . $merchant_id . '/deactivate', []);
        if (is_wp_error($response)) return $response;

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            return new WP_Error('agora_deactivate_failed', $body['error']['message'] ?? 'Deactivation failed');
        }

        return true;
    }

    /**
     * Notify Agora that a WC order's fulfillment status changed.
     */
    public function notify_order_status(string $agora_order_id, string $wc_status): true|WP_Error {
        $response = $this->post('/commerce/orders/' . $agora_order_id . '/wc-status', [
            'wc_status' => $wc_status,
        ]);
        if (is_wp_error($response)) return $response;

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            return new WP_Error('agora_notify_failed', $body['error']['message'] ?? 'Status notification failed');
        }

        return true;
    }

    // ─── Private HTTP helpers ─────────────────────────────────────────────────

    private function post(string $path, array $body, bool $require_auth = true): array|WP_Error {
        $headers = ['Content-Type' => 'application/json'];

        if ($require_auth) {
            if (empty($this->api_key)) {
                return new WP_Error('agora_not_connected', __('Store not connected to Agora. Go to WooCommerce → Settings → Agora.', 'agora-agent-commerce'));
            }
            $headers['Authorization'] = 'Bearer ' . $this->api_key;
        }

        return wp_remote_post(
            $this->base_url . $path,
            [
                'headers' => $headers,
                'body'    => wp_json_encode($body),
                'timeout' => 15,
            ]
        );
    }
}
