<?php
/**
 * Plugin Name:       Agora Agent Commerce
 * Plugin URI:        https://agora.sh/woocommerce
 * Description:       Connect your WooCommerce store to Agora OS. Let AI agents find and purchase your products on behalf of customers.
 * Version:           1.0.0
 * Requires at least: 6.0
 * Requires PHP:      8.0
 * Author:            Agora OS
 * Author URI:        https://agora.sh
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       agora-agent-commerce
 * Domain Path:       /languages
 * WC requires at least: 7.0
 * WC tested up to:   9.0
 */

defined('ABSPATH') || exit;

define('AGORA_PLUGIN_VERSION', '1.0.0');
define('AGORA_PLUGIN_FILE', __FILE__);
define('AGORA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AGORA_API_BASE_URL', 'https://api.agora.sh');

// Declare HPOS compatibility
add_action('before_woocommerce_init', function () {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

/**
 * Main plugin loader — only boots if WooCommerce is active.
 */
add_action('plugins_loaded', function () {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p><strong>Agora Agent Commerce</strong> requires WooCommerce to be installed and active.</p></div>';
        });
        return;
    }

    require_once AGORA_PLUGIN_DIR . 'includes/class-agora-api.php';
    require_once AGORA_PLUGIN_DIR . 'includes/class-agora-catalog.php';
    require_once AGORA_PLUGIN_DIR . 'includes/class-agora-checkout.php';
    require_once AGORA_PLUGIN_DIR . 'includes/class-agora-manifest.php';
    require_once AGORA_PLUGIN_DIR . 'includes/class-agora-webhook.php';
    require_once AGORA_PLUGIN_DIR . 'admin/class-agora-admin.php';

    new Agora_Admin();
    new Agora_Manifest();
    new Agora_Webhook();
    new Agora_Checkout();

    // Real-time product sync hooks
    add_action('woocommerce_update_product', 'agora_sync_product_hook', 10, 1);
    add_action('woocommerce_new_product',    'agora_sync_product_hook', 10, 1);
    add_action('woocommerce_delete_product', 'agora_delete_product_hook', 10, 1);
    add_action('woocommerce_variation_set_stock', 'agora_sync_variation_stock_hook', 10, 1);
});

function agora_sync_product_hook(int $product_id): void {
    $catalog = new Agora_Catalog();
    $catalog->sync_product($product_id);
}

function agora_delete_product_hook(int $product_id): void {
    // Products are removed from Agora on next catalog sync (mark in_stock=false)
    // Full delete endpoint not needed for Phase 0.5
}

function agora_sync_variation_stock_hook($variation): void {
    if (is_object($variation) && method_exists($variation, 'get_parent_id')) {
        agora_sync_product_hook($variation->get_parent_id());
    }
}

// ─── Activation / Deactivation ───────────────────────────────────────────────

register_activation_hook(__FILE__, function () {
    // Schedule daily catalog sync as a fallback to real-time hooks
    if (!wp_next_scheduled('agora_daily_catalog_sync')) {
        wp_schedule_event(time(), 'daily', 'agora_daily_catalog_sync');
    }
    // Flush rewrite rules so /.well-known/agent.json is accessible
    flush_rewrite_rules();
});

add_action('agora_daily_catalog_sync', function () {
    $catalog = new Agora_Catalog();
    $catalog->full_sync();
});

register_deactivation_hook(__FILE__, function () {
    wp_clear_scheduled_hook('agora_daily_catalog_sync');
});
