<?php
defined('ABSPATH') || exit;

/**
 * Agora admin settings page under WooCommerce → Settings → Agora.
 */
class Agora_Admin {

    public function __construct() {
        add_filter('woocommerce_settings_tabs_array', [$this, 'add_settings_tab'], 50);
        add_action('woocommerce_settings_tabs_agora',  [$this, 'render_settings_tab']);
        add_action('woocommerce_update_options_agora', [$this, 'save_settings']);

        // AJAX handlers for Connect and Disconnect buttons
        add_action('wp_ajax_agora_connect',    [$this, 'ajax_connect']);
        add_action('wp_ajax_agora_disconnect', [$this, 'ajax_disconnect']);
        add_action('wp_ajax_agora_sync_now',   [$this, 'ajax_sync_now']);

        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
    }

    public function add_settings_tab(array $tabs): array {
        $tabs['agora'] = __('Agora', 'agora-agent-commerce');
        return $tabs;
    }

    public function render_settings_tab(): void {
        require_once AGORA_PLUGIN_DIR . 'admin/views/settings-page.php';
    }

    public function save_settings(): void {
        // No WC settings fields stored via woocommerce_settings_* — all stored via AJAX
    }

    public function enqueue_scripts(string $hook): void {
        if ($hook !== 'woocommerce_page_wc-settings') return;
        if (!isset($_GET['tab']) || $_GET['tab'] !== 'agora') return;

        wp_enqueue_script(
            'agora-admin',
            plugins_url('assets/admin.js', AGORA_PLUGIN_FILE),
            ['jquery'],
            AGORA_PLUGIN_VERSION,
            true
        );

        wp_localize_script('agora-admin', 'agoraAdmin', [
            'ajaxUrl'  => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('agora_admin_nonce'),
            'strings'  => [
                'connecting'   => __('Connecting…', 'agora-agent-commerce'),
                'syncing'      => __('Syncing…', 'agora-agent-commerce'),
                'disconnecting'=> __('Disconnecting…', 'agora-agent-commerce'),
            ],
        ]);
    }

    // ─── AJAX: Connect ────────────────────────────────────────────────────────

    public function ajax_connect(): void {
        check_ajax_referer('agora_admin_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied.', 'agora-agent-commerce')], 403);
        }

        $api_key = sanitize_text_field($_POST['api_key'] ?? '');

        if (empty($api_key)) {
            wp_send_json_error(['message' => __('API key is required.', 'agora-agent-commerce')]);
        }

        // Validate the key against the Agora API
        $api    = new Agora_API();
        $result = $api->validate_key($api_key);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        // Store merchant ID and API key as WP options
        update_option('agora_merchant_id', sanitize_text_field($result['merchant_id'] ?? ''));
        update_option('agora_api_key',     $api_key);

        // Trigger initial catalog sync
        $catalog = new Agora_Catalog();
        $catalog->full_sync();

        // Flush rewrite rules so /.well-known/agent.json becomes active
        flush_rewrite_rules();

        wp_send_json_success([
            'merchant_id' => get_option('agora_merchant_id'),
            'message'     => __('Connected! Your catalog is syncing in the background.', 'agora-agent-commerce'),
        ]);
    }

    // ─── AJAX: Disconnect ────────────────────────────────────────────────────

    public function ajax_disconnect(): void {
        check_ajax_referer('agora_admin_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied.', 'agora-agent-commerce')], 403);
        }

        // Notify Agora so the merchant record is marked inactive (best-effort)
        $merchant_id = get_option('agora_merchant_id');
        if ($merchant_id) {
            $api = new Agora_API();
            $api->deactivate_merchant($merchant_id);
        }

        delete_option('agora_merchant_id');
        delete_option('agora_api_key');
        delete_option('agora_last_sync');
        delete_option('agora_last_sync_count');

        wp_send_json_success(['message' => __('Disconnected from Agora.', 'agora-agent-commerce')]);
    }

    // ─── AJAX: Manual Sync ───────────────────────────────────────────────────

    public function ajax_sync_now(): void {
        check_ajax_referer('agora_admin_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied.', 'agora-agent-commerce')], 403);
        }

        if (!get_option('agora_merchant_id')) {
            wp_send_json_error(['message' => __('Not connected to Agora.', 'agora-agent-commerce')]);
        }

        $catalog = new Agora_Catalog();
        $catalog->full_sync();

        wp_send_json_success([
            'last_sync'  => get_option('agora_last_sync'),
            'sync_count' => get_option('agora_last_sync_count'),
            'message'    => sprintf(
                /* translators: %d: number of synced products */
                __('Synced %d products.', 'agora-agent-commerce'),
                (int) get_option('agora_last_sync_count', 0)
            ),
        ]);
    }
}
